# Excavator boom control demo

This project is a simulation of a hypotethical control function of the boom motion on an excavator. The function is written in C with focus on function clarity rather than safety and reliablity. Therefore we do not recommend using the function in any control system.

![Excavator](images/Example%20excavator.png)

## Example system

To explain what this function does we can present the following block diagram.
![Simple function](images/Simple%20function.png)

Input to the control system is provided by an elactrical joystick attached to a standard analog input of 0 to 5V. The signal from the joystick is in range 0.5 to 4.5V. The output to the control valve is achieved by current control of two coils on a hydraulic valve. Range of the current is within 300 to 700 mA. Standard current control (valve control) ouputs are being used.

Here are some example modules that you might be looking at when developing such software:
- BODAS controller RC5-6/40
- IFM CR0708
- TTControl HY-TTC-32

Since the inputs and outputs never fit the whole range of the controller IOs we have introduced some parameters in our simple function. Taking into account those parameters we can acheve proper scaling and compensate for analog value fluctuations.

The control itself is proportionally applyig the joystick value to the appropriate output, taking into account the intended direction.
As there are many ways to solve this problem internaly we are not going to speculate on the best solution and we will just assume that the joystick is scaled in -100% to 100% internally, and when the values are negative the boom down coil is controlled.

## Simulating the system

In CppModel every simulation derrives from the CppModelBase::Simulation class. So our simulation will just extend that class.
There are three important parameters to set in the constructor of our new simulation. The name, which is later on used to identify simulation in CppModel Workspace, the simulation time, which decides for how long the simulation should be running, and step size which instructs CppModel how often to call our task method. This method is called RunCyclic and gets a single parameter, which holds the time at which it is being called.

The call to the controller function can happen in the RunCyclic method. Make sure to instruct the compiler if the code is writen in C rather than C++ as it is in this case.

After setting a value for the simulated input and providing context for the outputs we can also visualize those variables using the simulation inputs and ouputs properties.

BoomControlSimulation.cpp

```cpp
#include <cppmodel/Simulation.h>

extern "C" {
    #include "BoomControl.h"
}

class BoomControlSimulation : public CppModelBase::Simulation
{
    public:
    BoomControlSimulation()
    {
        name = "BoomControlSimulation";
        time = 10;
        stepSize = 0.01;
    }

    void RunCyclic(double stepTime) override
    {
        int joystickInput = (stepTime / time) * 4000 + 500;
        HydraulicValveOutput_ts outputContext;

        BoomControl(joystickInput, &outputContext);

        inputs["joystickInput"] = joystickInput;
        outputs["Up"] = outputContext.coilUp;
        outputs["Down"] = outputContext.coilDown;

    }
};
```

To run the simulation it is enough to instantiate this class and call it's Simulatie() method.
In the given example we do that by adding a main() function which will also allow us to compile this file as executable.

```cpp
int main(void)
{
    BoomControlSimulation simulation;

    simulation.Simulate();

    return 0;
}
```

This is the minimal simulation definition when using CppModel.

## Compiling

The downloadable example is provided with Windows libraries. To use another platform you can download the necessary libraries from https://workspace.cppmodel.com

### Windows Msys2-UCRT64

To compile the program use the following commands. Make sure to adjust it according to your library and code locations.

```shell
gcc -c .\controller\BoomControl.c
g++ -c .\BoomControlSimulation.cpp -I.\controller\ -I.\dependencies\include\ -DCPPHTTPLIB_OPENSSL_SUPPORT
g++ -o Runner.exe .\BoomControl.o .\BoomControlSimulation.o -L.\dependencies\lib\ -lCppModelBase -lssl -lcrypto -lcrypt32 -lws2_32
```

### Linux

To compile the program use the following commands. Make sure to adjust it according to your library and code locations.

```shell
gcc -c ./controller/BoomControl.c
g++ -c ./BoomControlSimulation.cpp -I./controller/ -I./dependencies/include/ -DCPPHTTPLIB_OPENSSL_SUPPORT
g++ -o Runner ./BoomControl.o ./BoomControlSimulation.o -L./dependencies/lib/ -lCppModelBase -lssl -lcrypto
```

## Running

Once compilation is done and the executable is ran for the first time there will be a login link which will guide you to your personal CppModel Workspace. The simulation can be located there by the name spcified in simulation's constructor and results can be observed.


![Results](images/Results.png)

## Code

Download this example from the following link https://download.cppmodel.com/SimpleBoomControl.zip

